<?php
namespace App\Controladores;

use App\Core\ControladorBase;
use App\Core\Sesion;
use App\Modelos\ReservasModel;
use MercadoPago\Config\MercadoPagoConfig;
use MercadoPago\Client\Payment\PaymentClient;
use MercadoPago\Exceptions\MPApiException;

class ReservasControlador extends ControladorBase {
    private $modelo;

    public function __construct(){
        parent::__construct();

        // Detección automática para usar credenciales TEST en localhost
        $isLocal = ($_SERVER['HTTP_HOST'] ?? '') === 'localhost' || ($_SERVER['HTTP_HOST'] ?? '') === '127.0.0.1';
        \MercadoPago\MercadoPagoConfig::setAccessToken(
            $isLocal
                ? 'TEST-2328678564240880-072317-fd312be8ae0d7c09d4c50e0d5ee88ef0-1361259632' // Sandbox
                : 'APP_USR-5967816942329157-072120-d1afd7bc5fa705746a889c6ac541a308-2545206547' // Producción
        );

        $this->modelo = new ReservasModel();
    }

    public function index(){
        $this->vista('Reservas/index');
    }

    public function estandar(){
        $this->vista('Reservas/estandar');
    }

    public function guardar(){
        if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !Sesion::verificarCsrf($_POST['csrf_token'] ?? '')) {
            http_response_code(400);
            exit('Solicitud inválida');
        }

        $datos = [
            'id_cliente' => $_SESSION['user_id'] ?? null,
            'tipo' => 'estandar',
            'fecha' => $_POST['fecha'],
            'hora' => $_POST['hora'],
            'personas' => (int) $_POST['personas'],
            'titular' => trim($_POST['titular']),
            'alergias' => trim($_POST['alergias'] ?? ''),
            'celebracion' => trim($_POST['celebracion'] ?? ''),
            'requerimientos_adicionales' => trim($_POST['requerimientos_adicionales'] ?? ''),
            'monto' => ((int) $_POST['personas']) * 50
        ];

        $resId = $this->modelo->crearReserva($datos);
        if (!$resId) {
            $_SESSION['error'] = "No se pudo crear la reserva.";
            return $this->redirect(BASE_URL . "reservas/estandar");
        }

        $this->modelo->registrarIngreso('estandar', $resId, $datos['monto']);
        $this->modelo->bloquearSlot($datos['fecha'], $datos['hora']);
        $codigo = $this->modelo->getCodigoById($resId);

        $_SESSION['success'] = "Reserva creada. Tu código de reserva es: {$codigo}";
        return $this->redirect(BASE_URL . "reservas/index");
    }

    public function ver(){
        $codigo = $_GET['codigo'] ?? '';
        $reserva = $this->modelo->getPorCodigo($codigo);
        $this->vista('Reservas/ver', ['reserva' => $reserva]);
    }

    public function actualizar(){
        $id = (int) $_POST['id_reserva'];
        $datos = [
            'fecha' => $_POST['fecha'],
            'hora' => $_POST['hora'],
            'personas' => (int) $_POST['personas'],
            'titular' => trim($_POST['titular']),
        ];
        $this->modelo->actualizarReserva($id, $datos);
        $this->redirect(BASE_URL . "reservas/ver?codigo=" . $_POST['codigo']);
    }

    public function eliminar(){
        $this->modelo->eliminarReserva((int) $_POST['id_reserva']);
        $_SESSION['success'] = "Reserva eliminada.";
        $this->redirect(BASE_URL . "reservas/index");
    }

    public function disponibilidad(){
        header('Content-Type: application/json');
        $fecha = $_GET['fecha'] ?? '';
        if (!$fecha) {
            echo json_encode([]);
            return;
        }
        try {
            $blocked = $this->modelo->getHorasNoDisponibles($fecha);
            echo json_encode($blocked);
        } catch (\PDOException $e) {
            http_response_code(500);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    public function procesar_pago_simulado() {
    header('Content-Type: application/json');

    // Leer el contenido JSON
    $input = json_decode(file_get_contents('php://input'), true);

    // Validar que se reciba el JSON correctamente
    if (!$input || !isset($input['email'], $input['tarjeta'], $input['exp_mes'], $input['exp_anio'], $input['cvv'], $input['datosReserva'])) {
        echo json_encode(['status' => 'error', 'detail' => 'Datos incompletos']);
        return;
    }

    // Validación básica de datos
    if (empty($input['email']) || empty($input['tarjeta']) || empty($input['exp_mes']) || empty($input['exp_anio']) || empty($input['cvv'])) {
        echo json_encode(['status' => 'error', 'detail' => 'Campos de tarjeta incompletos']);
        return;
    }

    // Extraer datos de reserva
    $res = $input['datosReserva'];
    if (empty($res['fecha']) || empty($res['hora']) || empty($res['personas'])) {
        echo json_encode(['status' => 'error', 'detail' => 'Datos de reserva incompletos']);
        return;
    }

    // Generar código de reserva simulado
    $codigo = 'RSV-' . strtoupper(uniqid());

    $datos = [
        'id_cliente'     => $_SESSION['user_id'] ?? null,
        'tipo'           => 'simulada',
        'fecha'          => $res['fecha'],
        'hora'           => $res['hora'],
        'personas'       => intval($res['personas']),
        'titular'        => $res['titular'] ?? '',
        'alergias'       => $res['alergias'] ?? '',
        'celebracion'    => $res['celebracion'] ?? '',
        'requerimientos_adicionales' => $res['req_extra'] ?? '',
        'email'          => $input['email'],
        'monto'          => floatval($input['monto'] ?? ($res['personas'] * 50)),
        'estado'         => 'confirmado',
        'codigo'         => $codigo,
        'fecha_registro' => date('Y-m-d H:i:s')
    ];

    // Guardar en la base de datos
    $ok = $this->modelo->guardarSimulacion($datos);

    if ($ok) {
        echo json_encode(['status' => 'success', 'codigo' => $codigo]);
    } else {
        echo json_encode(['status' => 'error', 'detail' => 'No se pudo guardar la reserva']);
    }
}

}
