// reserva.js

// ––––– Variables globales –––––
let hasInitMP     = false;
let bricksBuilder;
let currentStep   = 'main-options';
let selectedDate  = null;
let selectedTime  = null;
let currentMonth  = new Date().getMonth();
let currentYear   = new Date().getFullYear();
let calendarOpen  = false;
let mp;

// Meses en español
const months = [
    'Enero','Febrero','Marzo','Abril','Mayo','Junio',
    'Julio','Agosto','Septiembre','Octubre','Noviembre','Diciembre'
];

// Slots cada 30m de 08:00 a 22:30
const timeSlots = Array.from({ length: ((21 - 8) * 2) }, (_, i) => {
    const h = 8 + Math.floor(i / 2);
    const m = (i % 2) * 30;
    return `${h.toString().padStart(2, '0')}:${m.toString().padStart(2, '0')}`;
});

// ——— Helpers de UI ———
function showStep(stepId) {
    document.querySelectorAll('.step-block').forEach(el => el.style.display = 'none');
    document.getElementById('progress-bar').style.display = (stepId !== 'main-options' ? 'block' : 'none');
    const block = document.getElementById(stepId);
    if (block) block.style.display = 'block';
    currentStep = stepId;
    updateProgressIndicator(stepId);
    scrollToTop();
}

function updateProgressIndicator(stepId) {
    const raw = stepId.replace(/^step-/, '');
    const steps = Array.from(document.querySelectorAll('.progress-step'));
    const idx   = steps.findIndex(el => el.dataset.step === raw);
    const pct   = idx >= 0 ? ((idx + 1) / steps.length) * 100 : 0;
    const line  = document.getElementById('progress-line');
    if (line) line.style.setProperty('--progress-fill', pct + '%');
    steps.forEach((el, i) => {
        el.classList.toggle('active',    i === idx);
        el.classList.toggle('completed', i < idx);
    });
}

function scrollToTop() {
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

// ——— Pasos ———
function showRealizarReserva() { showStep('step-tipo'); }
function showVerReservas()      { showStep('ver-reservas'); }
function goBackToMain()         { showStep('main-options'); }

function toStep(step) {
    const id = step === 'tipo' ? 'step-tipo' : `step-${step}`;
    showStep(id);

    if (step === 'detalles') {
        generateCalendar();
        fetchUnavailableSlots(selectedDate || new Date());
    }

    if (step === 'resumen') {
        updateSummary();
        if (!hasInitMP) {
            console.log('🔧 Inicializando MercadoPago por primera vez');
            hasInitMP = true;
        } else {
            console.log('🔁 MercadoPago ya inicializado, salto initMercadoPago()');
        }
    }
}

// ——— Calendario ———
function toggleCalendar() {
    calendarOpen = !calendarOpen;
    document.getElementById('calendar-popup').classList.toggle('active', calendarOpen);
}

function previousMonth() {
    currentMonth = currentMonth === 0 ? 11 : currentMonth - 1;
    if (currentMonth === 11) currentYear--;
    generateCalendar();
}

function nextMonth() {
    currentMonth = currentMonth === 11 ? 0 : currentMonth + 1;
    if (currentMonth === 0) currentYear++;
    generateCalendar();
}

function generateCalendar() {
    const grid  = document.getElementById('calendar-grid');
    const title = document.getElementById('current-month');
    if (!grid || !title) return;
    grid.innerHTML = '';
    title.textContent = `${months[currentMonth]} ${currentYear}`;

    const firstDay = new Date(currentYear, currentMonth, 1).getDay();
    for (let i = 0; i < firstDay; i++) {
        const d = document.createElement('div');
        d.className = 'calendar-day disabled';
        grid.appendChild(d);
    }

    const daysInMonth = new Date(currentYear, currentMonth + 1, 0).getDate();
    const todayTS = new Date().setHours(0, 0, 0, 0);

    for (let d = 1; d <= daysInMonth; d++) {
        const cell = document.createElement('div');
        cell.className = 'calendar-day';
        cell.textContent = d;
        const thisDate = new Date(currentYear, currentMonth, d);

        if (thisDate.getTime() < todayTS) {
            cell.classList.add('disabled');
        } else {
            cell.addEventListener('click', () => selectDate(thisDate));
            if (selectedDate && thisDate.toDateString() === selectedDate.toDateString()) {
                cell.classList.add('selected');
            }
        }
        if (thisDate.toDateString() === new Date().toDateString()) {
            cell.classList.add('today');
        }
        grid.appendChild(cell);
    }
}

function selectDate(dt) {
    selectedDate = dt;
    document.getElementById('selected-date').textContent = dt.toLocaleDateString('es-ES', {
        weekday: 'long', year: 'numeric', month: 'long', day: 'numeric'
    });
    toggleCalendar();
    clearFieldError('date-input');
    fetchUnavailableSlots(dt);
}

function fetchUnavailableSlots(dateObj) {
    const iso = dateObj.toISOString().split('T')[0];
    const url = `${window.BASE_URL}reservas/disponibilidad?fecha=${iso}`;
    console.log('👉 Disponibilidad URL:', url);

    fetch(url)
        .then(r => r.ok ? r.json() : Promise.reject(`Status ${r.status}`))
        .then(data => generateTimeSlots(data))
        .catch(err => {
            console.warn('⚠ fetchUnavailableSlots error:', err);
            generateTimeSlots([]);
        });
}

function generateTimeSlots(unavailable = []) {
    const cont = document.getElementById('time-selector');
    cont.innerHTML = '';
    timeSlots.forEach(t => {
        const slot = document.createElement('div');
        slot.className = 'time-slot';
        slot.textContent = t;
        if (unavailable.includes(t)) {
            slot.classList.add('disabled');
        } else {
            slot.addEventListener('click', () => selectTime(t, slot));
        }
        cont.appendChild(slot);
    });
}

function selectTime(t, el) {
    selectedTime = t;
    document.querySelectorAll('.time-slot').forEach(s => s.classList.remove('selected'));
    el.classList.add('selected');
    clearFieldError('time-selector');
}

// ——— Personas ———
function increasePersonas() {
    const inp = document.getElementById('personas');
    const v   = parseInt(inp.value) || 1;
    if (v < parseInt(inp.max)) inp.value = v + 1;
}
function decreasePersonas() {
    const inp = document.getElementById('personas');
    const v   = parseInt(inp.value) || 1;
    if (v > parseInt(inp.min)) inp.value = v - 1;
}

// ——— Validación ———
function validateAndContinue() {
    const errs = [];
    if (!selectedDate)    errs.push({ field: 'date-input',    message: 'Selecciona una fecha' });
    if (!selectedTime)    errs.push({ field: 'time-selector', message: 'Selecciona una hora' });
    const p = document.getElementById('personas').value;
    if (!p || parseInt(p) < 1) errs.push({ field: 'personas', message: 'Número de personas inválido' });
    const tit = document.getElementById('titular').value.trim();
    if (!tit) errs.push({ field: 'titular', message: 'Ingresa el nombre del titular' });

    if (errs.length) {
        showErrors(errs);
        scrollToTop();
    } else {
        hideErrors();
        toStep('preferencias');
    }
}

function showErrors(errs) {
    const ec = document.getElementById('error-messages');
    ec.innerHTML = '';
    errs.forEach(e => {
        const d = document.createElement('div');
        d.className = 'error-message';
        d.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${e.message}`;
        ec.appendChild(d);
        if (e.field) {
            const fg = document.getElementById(e.field).closest('.form-group');
            fg && fg.classList.add('error');
        }
    });
    ec.style.display = 'block';
}

function hideErrors() {
    document.getElementById('error-messages').style.display = 'none';
    document.querySelectorAll('.form-group').forEach(g => g.classList.remove('error'));
}

function clearFieldError(id) {
    const grp = document.getElementById(id).closest('.form-group');
    grp && grp.classList.remove('error');
}

// ——— Resumen ———
function updateSummary() {
    const cont = document.querySelector('.resumen-card');
    cont.innerHTML = '';
    const data = {
        fecha:        selectedDate,
        hora:         selectedTime,
        personas:     document.getElementById('personas').value,
        titular:      document.getElementById('titular').value,
        alergias:     document.getElementById('alergias').value     || 'Ninguna',
        celebracion:  document.getElementById('celebracion').value || 'Ninguna',
        req:          document.getElementById('requerimientos_adicionales').value || 'Ninguno'
    };

    function addItem(icon, label, val) {
        const d = document.createElement('div');
        d.className = 'resumen-item';
        d.innerHTML = `
        <span class="resumen-label"><i class="${icon}"></i> ${label}:</span>
        <span class="resumen-value">${val}</span>
        `;
        cont.appendChild(d);
    }

    addItem('fas fa-calendar', 'Fecha', data.fecha.toLocaleDateString('es-ES',{ weekday:'long',year:'numeric',month:'long',day:'numeric'}));
    addItem('fas fa-clock','Hora', data.hora);
    addItem('fas fa-users','Personas', data.personas);
    addItem('fas fa-user','Titular', data.titular);
    addItem('fas fa-exclamation-triangle','Alergias', data.alergias);
    addItem('fas fa-birthday-cake','Celebración', data.celebracion);
    addItem('fas fa-comment','Requerimientos', data.req);

    const total = 50 * parseInt(data.personas, 10);
    document.getElementById('precio-estimado').textContent = total.toFixed(2);
}

function obtenerResumenReserva() {
    return {
        fecha: selectedDate ? selectedDate.toISOString().split('T')[0] : '',
        hora: selectedTime || '',
        personas: parseInt(document.getElementById('personas').value) || 1,
        titular: document.getElementById('titular').value.trim(),
        alergias: document.getElementById('alergias').value.trim(),
        celebracion: document.getElementById('celebracion').value.trim(),
        requerimientos: document.getElementById('requerimientos_adicionales').value.trim()
    };
}


function setupSimulacionPago() {
    const btn = document.getElementById('btn-simular-pago');
    if (btn) btn.onclick = simularPago;
}

function simularPago() {
    const email = document.getElementById('email-simulado').value.trim();
    const card = document.getElementById('fake-card').value.trim();
    const name = document.getElementById('fake-name').value.trim();
    const exp = document.getElementById('fake-exp').value.trim();
    const cvv = document.getElementById('fake-cvv').value.trim();

    // Validar campos obligatorios
    if (!email || !card || !name || !exp || !cvv) {
        alert("Por favor, completa todos los campos para simular el pago.");
        return;
    }

    // Validar formato MM/YY
    const match = exp.match(/^(\d{2})\/(\d{2})$/);
    if (!match) {
        alert("Fecha de expiración inválida. Usa formato MM/YY.");
        return;
    }

    const [ , monthStr, yearStr ] = match;
    const month = parseInt(monthStr, 10);
    const year = parseInt("20" + yearStr); // convierte '25' => 2025

    // Validar mes
    if (month < 1 || month > 12) {
        alert("Mes inválido. Usa un valor entre 01 y 12.");
        return;
    }

    // Validar si la tarjeta ya expiró
    const now = new Date();
    const currentMonth = now.getMonth() + 1;
    const currentYear = now.getFullYear();

    if (year < currentYear || (year === currentYear && month < currentMonth)) {
        alert("La tarjeta ha expirado.");
        return;
    }

    // Obtener resumen de reserva
    const resumen = obtenerResumenReserva();
    const monto = resumen.personas * 50;

    const payload = {
        email,
        tarjeta: card,
        nombre: name,
        exp_mes: monthStr,
        exp_anio: yearStr,
        cvv,
        monto,
        datosReserva: resumen
    };

    fetch(BASE_URL + 'reservas/procesar_pago', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
    })
    .then(r => r.json())
    .then(data => {
        if (data.status === 'success') {
            document.getElementById('codigo-reserva').textContent = data.codigo;
            toStep('pago');
        } else {
            alert("❌ No se pudo registrar la reserva. Intenta nuevamente.");
        }
    })
    .catch(err => {
        console.error("Error en la simulación:", err);
        alert("❌ Ocurrió un error inesperado.");
    });
}


// ——— Bootstrapping ———
document.addEventListener('DOMContentLoaded', () => {
    showStep('main-options');
    setupSimulacionPago(); // ✅ Agregado aquí

    document.addEventListener('click', e => {
        const pop = document.getElementById('calendar-popup');
        const inp = document.getElementById('date-input');
        if (calendarOpen && pop && !pop.contains(e.target) && !inp.contains(e.target)) {
            toggleCalendar();
        }
    });
});



document.addEventListener('DOMContentLoaded', () => {
    const expInput = document.getElementById('fake-exp');

    expInput.addEventListener('input', function (e) {
        let value = this.value.replace(/\D/g, ''); // solo números
        if (value.length >= 3) {
            value = value.substring(0, 2) + '/' + value.substring(2, 4);
        }
        this.value = value;
    });

});
